import Config from "@jangaroo/runtime/Config";
import StudioStartupPlugin from "@coremedia/studio-client.ext.startup-plugins/StudioStartupPlugin";
import PluginRules from "@coremedia/studio-client.ext.ui-components/PluginRules";
import DocumentTabPanel from "@coremedia/studio-client.main.editor-components/sdk/premular/DocumentTabPanel";
import NestedRulesPlugin from "@coremedia/studio-client.ext.ui-components/plugins/NestedRulesPlugin";
import OnlyIf from "@coremedia/studio-client.main.editor-components/sdk/plugins/OnlyIf";
import DocumentForm from "@coremedia/studio-client.main.editor-components/sdk/premular/DocumentForm";
import AddItemsPlugin from "@coremedia/studio-client.ext.ui-components/plugins/AddItemsPlugin";
import PropertyFieldGroup from "@coremedia/studio-client.main.editor-components/sdk/premular/PropertyFieldGroup";
import type {
  PropertyFieldConfig,
  AddCmArticlePropertyFieldExtension,
} from "@coremedia-internal/studio-client.main.example-microfrontend-module-federation";
import { as, joo } from "@jangaroo/runtime";
import type { ValueExpression } from "@coremedia/studio-client.client-core";
import { Content } from "@coremedia/studio-client.cap-rest-client";
import WebComponent from "./WebComponent";

async function loadNewPlugin(apiConfig: AddCmArticlePropertyFieldExtension): Promise<void> {
  const plugin = await joo.pluginLoader.loadPlugin({
    name: "example_microfrontend_studio_plugin",
    path: new URL(
      "/packages/coremedia-internal__studio-client.main.example-microfrontend-extjs/example-microfrontend-module-federation/appEntry.js",
      window.location.href,
    ),
  });
  const rootModule = (await plugin.getModuleFactory("."))?.();
  if (
    rootModule &&
    typeof rootModule === "object" &&
    "initPlugin" in rootModule &&
    typeof rootModule.initPlugin === "function"
  ) {
    await rootModule.initPlugin(apiConfig);
  }
}

interface ExampleMicrofrontendStudioPluginConfig extends Config<StudioStartupPlugin> {}

class ExampleMicrofrontendStudioPlugin extends StudioStartupPlugin {
  declare Config: ExampleMicrofrontendStudioPluginConfig;

  constructor(config: Config<ExampleMicrofrontendStudioPlugin>) {
    super(config);
  }

  override async init(): Promise<void> {
    await super.init();
    await loadNewPlugin({
      addCmArticlePropertyField: addCmArticlePropertyField,
    });
  }
}

const addCmArticlePropertyField = (fieldConfig: PropertyFieldConfig): void => {
  new PluginRules({
    rules: [
      Config(DocumentTabPanel, {
        plugins: [
          Config(OnlyIf, {
            condition: (cmp) => cmp.itemId === "CMArticle",
            then: Config(NestedRulesPlugin, {
              rules: [
                Config(DocumentForm, {
                  itemId: "contentTab",
                  plugins: [
                    Config(AddItemsPlugin, {
                      index: 0,
                      items: [
                        Config(PropertyFieldGroup, {
                          itemId: fieldConfig.id,
                          title: fieldConfig.label,
                          items: [
                            Config(WebComponent, {
                              webComponentFactory: fieldConfig.factory,
                              listeners: {
                                // for Studio tab-reuse: always update the uuid attribute of the embedded web component when bindTo changes
                                // the ExtJS WebComponent does have the bindTo because it gets automatically passed through by the PropertyFieldGroup
                                afterrender: (webCmp: WebComponent & { bindTo: ValueExpression<Content> }) => {
                                  async function updateContent() {
                                    const webElement = webCmp.getWebElement();
                                    if (!webElement) {
                                      return;
                                    }
                                    const content = as(webCmp.bindTo.getValue(), Content);
                                    if (content) {
                                      const loadedContent = await content.load();
                                      webElement.setAttribute("uuid", loadedContent.getUuid());
                                    } else {
                                      webElement.removeAttribute("uuid");
                                    }
                                  }

                                  webCmp.bindTo.addChangeListener(updateContent);

                                  webCmp.on("beforedestroy", () => webCmp.bindTo.removeChangeListener(updateContent));

                                  updateContent();
                                },
                              },
                            }),
                          ],
                        }),
                      ],
                    }),
                  ],
                }),
              ],
            }),
          }),
        ],
      }),
    ],
  });
};

export default ExampleMicrofrontendStudioPlugin;
