import Config from "@jangaroo/runtime/Config";
import ConfigUtils from "@jangaroo/runtime/ConfigUtils";
import Component from "@jangaroo/ext-ts/Component";
import { EventUtil } from "@coremedia/studio-client.client-core";

interface WebComponentConfig extends Config<Component>, Partial<Pick<WebComponent, "webComponentFactory">> {
  updateLayoutOnResize?: boolean;
}

class WebComponent extends Component {
  declare Config: WebComponentConfig;

  webComponentFactory?: () => HTMLElement;

  #webComponent?: HTMLElement | undefined;

  updateLayoutOnResize = true;

  #resizeObserver: ResizeObserver;

  constructor(config: Config<WebComponent>) {
    super(ConfigUtils.apply(Config(WebComponent), config));
    this.#resizeObserver = new ResizeObserver(() => {
      // Not 100% sure why this is needed, but we had problems with updateLayout() in the past when it was not executed in the event perso cycle.
      // In this case, we get an "ResizeObserver perso completed with undelivered notifications" error otherwise.
      EventUtil.invokeLater(() => this.updateLayout());
    });
  }

  override afterRender(): void {
    if (!this.webComponentFactory) {
      return;
    }
    this.#webComponent = this.webComponentFactory();
    this.#webComponent.style.display = "block";
    this.el.dom.appendChild(this.#webComponent);
    if (this.updateLayoutOnResize) {
      this.#resizeObserver.observe(this.#webComponent);
    }
  }

  override beforeDestroy(): void {
    this.#resizeObserver.disconnect();
  }

  getWebElement(): HTMLElement | null {
    return this.#webComponent ?? null;
  }
}

export default WebComponent;
